create or replace package body tbicds.PCK_CPA is

/* Copyright 2015 Intellica Corporation 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at
    http://www.apache.org/licenses/LICENSE-2.0
Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
/******
delete an event
******/  
procedure DeleteEvent (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_vPatientID             in varchar2,
      pi_nPatPWEventID          in number,
    
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2
      )
is
    v_nTemp number := 0;
    
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    delete from pat_pw_event_entry t 
    where t.pat_pw_event_id = pi_nPatPWEventID;
    commit;
    
    delete from pat_pw_event_module t 
    where t.pat_pw_event_id = pi_nPatPWEventID;
    commit;
    
    delete from pat_pw_event t 
    where t.pat_pw_event_id = pi_nPatPWEventID;
    commit;
    
    update pat_pw_event set
      plan_pat_pw_event_id = null
    where plan_pat_pw_event_id = pi_nPatPWEventID;
    commit;
    
exception
    when others then
      rollback;
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;



/******
Check if the user can delete the event
******/  
procedure AllowEventDelete (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_vPatientID             in varchar2,
      pi_nPatPWEventID          in number,
         
      po_nAllowDelete           out number,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2
      )
is
    v_nTemp number := 0;
    
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    po_nAllowDelete := 1;
   
    --check the status 2 = event is closed
    select t.status into v_nTemp 
    from pat_pw_event t
    where t.pat_pw_event_id = pi_nPatPWEventID;
    if v_nTemp = 2 then
       po_nAllowDelete := 0;
       return;  
    end if;
    
    --check if the event has any completed instruments
    select count(*) into v_nTemp  
    from pat_pw_event_module t
    where t.pat_pw_event_id = pi_nPatPWEventID
    and nvl(t.intake_id,0) > 0;
    if v_nTemp > 0 then
       po_nAllowDelete := 0;
       return;  
    end if;

    --check if the event has any note text
    select count(*) into v_nTemp  
    from pat_pw_event t
    where t.pat_pw_event_id = pi_nPatPWEventID
    and t.note_text is not null
    and length( nvl(t.note_text,'') ) > 0;
    if v_nTemp > 0 then
       po_nAllowDelete := 0;
       return;  
    end if;

exception
    when others then
      rollback;
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

/****
insert a new pat pw event
***/
procedure InsertPatPWEvent (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      
      pi_nRegionID              in number,
      pi_nSiteID                in number,
      
      pi_vPatientID             in varchar2,
      pi_nPatPWID               in number,
   
      pi_vEventTitle            in varchar2,
      pi_dtEventDate            in date,
      pi_nAdjustFuture          in number,
      
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2
      )
is
      v_nPatPWEventID number := 0;
   
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --get a new patient pathway event id
    select SEQPATPWEVENTID.nextval into v_nPatPWEventID from dual;
    
    --insert the event 
    insert into pat_pw_event
                (PATIENT_ID,
                 STATUS,
                 DATE_SCHEDULED,
                 DATE_STARTED,
                 DATE_COMPLETED,
                 REGION_ID,
                 SITE_ID,
                 PAT_PW_ID,
                 EVENT_TITLE,
                 EVENT_DESCR,
                 SORT_ORDER,
                 EVENT_FREQUENCY,
                 PW_EVENT_TYPE_ID,
                 PAT_PW_EVENT_ID,
                 CREATED_BY)
        values(pi_vPatientID,
               0,
               pi_dtEventDate,
               null,
               null,
               pi_nRegionID,
               pi_nSiteID,
               pi_nPatPWID,
               pi_vEventTitle,
               '',
               99,
               null,
               1, --scheduled
               v_nPatPWEventID,
               pi_nUserID);
        
        commit;                  
        
        --update the event just created to allow for adjust
        if pi_nAdjustFuture = 1 then
          
          UpdatePatPWEvent( pi_vSessionID,
                            pi_vSessionClientIP,
                            pi_nUserID,
                            pi_vPatientID,
                            v_nPatPWEventID,
                            
                            pi_vEventTitle,
                            pi_dtEventDate,
                            pi_nAdjustFuture,
                            
                            po_nStatusCode,
                            po_vStatusComment);
          commit;
        
        end if;
        
exception
    when others then
      rollback;
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

/***
Update the pw event info
****/
procedure UpdatePatPWEvent (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_vPatientID             in varchar2,
      pi_nPatPWEventID          in number,
      
      pi_vEventTitle            in varchar2,
      pi_dtEventDate            in date,
      pi_nAdjustFuture          in number,
      
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2
      )
is

      v_nIndex number := 0;
      
      v_dtOldScheduled date := null;
      v_nPatPWID number := 0;
      v_nDays number := 0;
      v_nEventModuleID number := 0;
   
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    v_nEventModuleID := 0;
    
    --keep the original date for calculations
    select date_scheduled, pat_pw_id 
    into v_dtOldScheduled, v_nPatPWID
    from pat_pw_event 
    where pat_pw_event_id = pi_nPatPWEventID
    and patient_id = pi_vPatientID;
    
    --update title and date
    update pat_pw_event 
    set event_title = pi_vEventTitle,
        date_scheduled = pi_dtEventDate
    where pat_pw_event_id = pi_nPatPWEventID
    and patient_id = pi_vPatientID;
    commit;
        
    --get the number of days this date is different    
    if pi_nAdjustFuture = 1 then
       
       --get the date diff 
       SELECT trunc(pi_dtEventDate)- (v_dtOldScheduled) 
         into v_nDays 
         FROM DUAL;
             
       --update all events after this event
       /*update pat_pw_event pe
          set pe.date_scheduled =  pe.date_scheduled + v_nDays
        where patient_id = pi_vPatientID
          --and pe.pat_pw_event_id > pi_nPatPWEventID
          and pe.date_scheduled > pi_dtEventDate 
          and pe.date_completed is null 
          and pe.pat_pw_id = pat_pw_id; */
       update pat_pw_event pe
          set pe.date_scheduled =  pe.date_scheduled + v_nDays
        where patient_id = pi_vPatientID
          and pe.pat_pw_event_id > pi_nPatPWEventID
          and pe.pat_pw_id = pat_pw_id;
       commit;         
      
    end if;
       
    commit;

exception
    when others then
      rollback;
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;



/*delete a scheduled module*/
procedure DeletePatPWEventModules (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      
      pi_vPatientID             in varchar2,
      pi_nPatPWEventID          in number,
      pi_vMIDS                  in varchar2,
     
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2
      )
is

      v_nIndex number := 0;
      v_vMIDPiece varchar2(4000) := ''; 
      v_nMID number := 0;   
      v_nModuleFor number := 2;
      
      v_dtOldScheduled date := null;
      v_nPatPWID number := 0;
      v_nDays number := 0;
      v_nEventModuleID number := 0;
   
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    v_nEventModuleID := 0;
    
     --loop and add modules
     loop
      
      --get one mid   
      v_vMIDPiece := pck_common.GetPiece(pi_vMIDs, ',' , v_nIndex);
      
      EXIT WHEN v_vMIDPiece is null;
            
      begin
        
           --get the mid and bump the index
           v_nMID := to_number(v_vMIDPiece);
           v_nIndex := v_nIndex +1;
           
           --only delete modules that do not have and intake_id
           delete from pat_pw_event_module t
           where t.patient_id = pi_vPatientID 
           and t.pat_pw_event_id = pi_nPatPWEventID 
           and nvl(t.intake_id,0) <= 0 
           and t.mid = v_nMID;
                                    
           commit;                                 
      
      exception
      when others then
           null;
      end;
      
     end loop;
       
    commit;

exception
    when others then
      rollback;
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

/*****
Gets all pathway event modules associated 
with a specific patient pathway event
******/
procedure GetPWEventModuleRS (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_vPatientID             in varchar2,
      pi_nPatPWEventID             in number,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2,
      rs                        out RetRefCursor
      )
is

     v_vSQL varchar2(2000) :=
      'SELECT t.*,
              t3.module, 
              t3.description, 
              t3.module_alt_lang, 
              t3.description_alt_lang, 
              t3.acronym, 
              t3.score_logic,
              
              (select nvl(tt.status,0) from
                pat_pw_event tt
                where tt.pat_pw_event_id = :PAT_PW_EVENT_ID) 
                as event_status     
              
       FROM tbicds.PAT_PW_EVENT_MODULE t,
            tbicds.intake_module t3 
            
       WHERE t.patient_id = :PATIENT_ID 
         and t.PAT_PW_EVENT_ID = :PW_EVENT_ID 
         and t3.mid = t.mid
      
       ORDER BY t.sort_order ASC';
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --open the result set
     open rs 
      for v_vSQL
    using pi_nPatPWEventID,
          pi_vPatientID,
          pi_nPatPWEventID;
    
    commit;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

/***
insert modules into the pat pw event module list
****/
procedure InsertPatPWEventModules (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      
      pi_vPatientID             in varchar2,
      pi_nPatPWEventID          in number,
      pi_vMIDS                  in varchar2,
      pi_vModuleFors            in varchar2,
     
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2
      )
is

      v_nIndex number := 0;
      v_vMIDPiece varchar2(4000) := ''; 
      v_vFORPiece varchar2(4000) := '';  
      v_nMID number := 0;   
      v_nModuleFor number := 2;
      
      v_dtOldScheduled date := null;
      v_nPatPWID number := 0;
      v_nDays number := 0;
      v_nEventModuleID number := 0;
   
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    v_nEventModuleID := 0;
    
     --loop and add modules
     loop
      
      --get one mid   
      v_vMIDPiece := pck_common.GetPiece(pi_vMIDs, ',' , v_nIndex);
      
      --get one module for
      v_vFORPiece := pck_common.GetPiece(pi_vModuleFors, ',' , v_nIndex);
      
      EXIT WHEN v_vMIDPiece is null;
            
      begin
        
           --get the mid and bump the index
           v_nMID := to_number(v_vMIDPiece);
           v_nModuleFor := to_number(v_vFORPiece);
           v_nIndex := v_nIndex +1;
           
           select SEQPATPWEVENTMODULEID.Nextval 
           into v_nEventModuleID from dual;
           
           insert into 
           pat_pw_event_module(PATIENT_ID,
                               MID,
                               STATUS,
                               REGION_ID,
                               SITE_ID,
                               DATE_STARTED,
                               DATE_COMPLETED,
                               PAT_PW_ID,
                               INTAKE_ID,
                               MODULE_FOR,
                               SORT_ORDER,
                               PAT_PW_EVENT_ID,
                               DATE_SCHEDULED,
                               PW_EVENT_MODULE_ID)
            values(pi_vPatientID,
                   
                   v_nMID,
                   
                   0,--status
                   
                   (select REGION_ID 
                    from pat_pw_event 
                    where pat_pw_event_id = pi_nPatPWEventID),
                   
                    (select SITE_ID 
                    from pat_pw_event 
                    where pat_pw_event_id = pi_nPatPWEventID),
                    
                    null, --date started
                    
                    null, --date completed
                   
                   (select PAT_PW_ID 
                    from pat_pw_event 
                    where pat_pw_event_id = pi_nPatPWEventID),
                   
                    null, --intake_id
                    
                    v_nModuleFor, --module for
                    
                    (select nvl(max(sort_order), 0) + 1 
                       from pat_pw_event_module 
                      where pat_pw_event_id = pi_nPatPWEventID),
                      
                    pi_nPatPWEventID,
                    
                    sysdate,
                    
                    v_nEventModuleID);
                                    
           commit;                                 
           
      
      exception
      when others then
           null;
      end;
      
     end loop;
       
    commit;

exception
    when others then
      rollback;
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;


/*****
Gets all clinical program area types
******/
procedure GetCPATypeRS (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2,
      rs                        out RetRefCursor
      )
is

     v_vSQL varchar2(500) :=
      'SELECT t.*
       FROM tbicds.CPA_Type t 
       ORDER BY upper(t.sort_order) ASC';
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --open the result set
    open rs for v_vSQL;
    commit;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;


/*****
Gets all clinical program areas by region/site
******/
procedure GetCPARS (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_nRegionID              in number,
      pi_nSiteID                in number,
      
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2,
      rs                        out RetRefCursor
      )
is

     v_vSQL varchar2(500) :=
      'SELECT t.*
       FROM tbicds.CPA t,
            tbicds.CPA_TYPE t2
       WHERE t.region_id = :REGION_ID
       and t.site_id = :SITE_ID 
       and t.CPA_TYPE_ID = t2.CPA_TYPE_ID 
       ORDER BY t2.sort_order ASC';
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --open the result set
    open rs 
    for v_vSQL
      using pi_nRegionID, pi_nSiteID;
    commit;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

/*****
Gets all pathways for a clinical program area 
each site has its own list of clinical program areas
******/
procedure GetCPAPWRS (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_nCPAID                 in number,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2,
      rs                        out RetRefCursor
      )
is

     v_vSQL varchar2(500) :=
      'SELECT t.*
       FROM tbicds.CPA_PW t 
       WHERE CPA_ID = :CPA_ID
       ORDER BY upper(t.PW_TITLE) ASC';
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --open the result set
    open rs 
    for v_vSQL
      using pi_nCPAID;
    commit;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

/*****
Gets all pathway events for a pathway
******/
procedure GetPWEventRS (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_nPWID                  in number,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2,
      rs                        out RetRefCursor
      )
is

     v_vSQL varchar2(500) :=
      'SELECT t.*
       FROM tbicds.CPA_PW_EVENT t 
       WHERE PW_ID = :PW_ID
       ORDER BY t.sort_order ASC';
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --open the result set
    open rs 
    for v_vSQL
      using pi_nPWID;
    commit;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;


/*****
Gets all modules for a pathway event
******/
procedure GetPWEventModuleRS (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_nPWID                  in number,
      pi_nPWEventID             in number,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2,
      rs                        out RetRefCursor
      )
is

     v_vSQL varchar2(500) :=
      'SELECT t.*, 
              t2.module, 
              t2.description, 
              t2.module_alt_lang, 
              t2.description_alt_lang, 
              t2.acronym, 
              t2.score_logic 
       FROM tbicds.CPA_PW_EVENT_MODULE t,
            tbicds.intake_module t2 
       WHERE PW_ID = :PW_ID
       and PW_EVENT_ID = :PW_EVENT_ID
       and t.mid = t2.mid
       ORDER BY t.sort_order ASC';
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --open the result set
    open rs 
    for v_vSQL
      using pi_nPWID,
            pi_nPWEventID;
    commit;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;


/*****
Gets all events and modules for a pathway 
******/
procedure GetPWEventAndModuleRS (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_nPWID                  in number,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2,
      rs                        out RetRefCursor
      )
is

     v_vSQL varchar2(500) :=
      'select * 
        from cpa_pw_event t1,
             cpa_pw_event_module t2,
             intake_module t3
        where t1.cpa_id = t2.cpa_id(+)
          and t1.pw_event_id = t2.pw_event_id(+)
          and t1.pw_id = t2.pw_id(+)
          and t2.mid = t3.mid(+)
          and t1.pw_id = :PW_ID   
        order by t1.sort_order ';
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --open the result set
    open rs 
    for v_vSQL
      using pi_nPWID;
    commit;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;




----USER tables------------------------------------------------------

/*****
update a users default clinical program area
******/
procedure UpdateUserDefaultCPA (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_nCPAID                 in number,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2
      )
is

     v_vSQL varchar2(500) :=
      'UPDATE tbicds.fx_user 
       set default_cpa_id = :cpa_id
       where fx_user_id = :user_id';
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --execute the sql
    execute immediate v_vSQL 
                using pi_nUserID,
                      pi_nCPAID;
    
    commit;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

/*****
Gets a users clinical program areas
******/
procedure GetUserCPARS (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2,
      rs                        out RetRefCursor
      )
is

     v_vSQL varchar2(1024) :=
      'SELECT t.*,
              
              (select count(*) 
                 from tbicds.fx_user
                where default_cpa_id = t.cpa_id
                and fx_user_id = :FX_USER_ID) as default_cpa_id
                
         FROM tbicds.CPA t
         
        where t.region_id = (select region_id 
                            from tbicds.fx_user 
                            where fx_user_id = :FX_USER_ID)
                             
          and t.site_id = (select site_id 
                            from tbicds.fx_user 
                            where fx_user_id = :FX_USER_ID)
                            
       ORDER BY upper(t.cpa_title) ASC';
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --open the result set
    open rs 
    for v_vSQL 
    using pi_nUserID,
          pi_nUserID,
          pi_nUserID;
    
    commit;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;
 
-------patient tables ----------------------------------------------


/*****
Gets all pathways associated with a patient, 
ordered by region, site and clinical program area
******/
procedure GetPatPWRS (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_vPatientID             in varchar2,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2,
      rs                        out RetRefCursor
      )
is

     v_vSQL varchar2(500) :=
        'SELECT t.*
       FROM tbicds.PAT_PW t
       WHERE t.patient_id = :PATIENT_ID 
       ORDER BY t.region_id, 
                t.site_id,
                t.date_created DESC';
       
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --open the result set
     open rs 
      for v_vSQL
    using pi_vPatientID;
    
    commit;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

/*Check if the Event is Read-Only*/
procedure IsPatPWEventReadOnly (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_vPatientID             in varchar2,
      pi_nPatPWID               in number,
      pi_nPatPWEventID          in number,
      po_nIsReadOnly            out number, --0 = False = Is Not Read Only; 1 = True = Is Read Only 
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2
      )
is

  v_nIsReadOnly number := 1; --TRUE

begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    begin
      select 
        case nvl(t.status,0) 
          when 2 then 1 --TRUE = Is Read Only
          else 0 --FALSE = Is Not Read Only
        end            
        into v_nIsReadOnly
      from pat_pw_event t
      where t.patient_id = pi_vPatientID
        and t.pat_pw_id = pi_nPatPWID
        and t.pat_pw_event_id = pi_nPatPWEventID;
    exception
      when others then
        v_nIsReadOnly := 1;
    end;
    
    po_nIsReadOnly := v_nIsReadOnly;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

/*****
Gets all pathway events for a specific patient pathway
******/
procedure GetPatPWEventRS (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_vPatientID             in varchar2,
      pi_nPatPWID                  in number,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2,
      rs                        out RetRefCursor
      )
is

     v_vSQL varchar2(4000) :=
      'SELECT  t.PATIENT_ID,	
                t.STATUS,
                t.DATE_SCHEDULED,
                t.DATE_STARTED,
                t.DATE_COMPLETED,
                t.REGION_ID,
                t.SITE_ID,
                t.PAT_PW_ID,
                t.PLAN_TEXT,
                t.PLAN_PAT_PW_EVENT_ID,
               
       decode(trim(event_title),
                                 null,

                decode(t.date_completed,
                       null,
                       '''' || to_char(date_scheduled, ''MM/DD/YYYY'' ), 
                       '''' || to_char(date_completed, ''MM/DD/YYYY'')
                       ),
                       
                decode(t.date_completed,
                       null,
                       '''' || to_char(date_scheduled, ''MM/DD/YYYY'') || '' - ('' || event_title || '')'', 
                       '''' || to_char(date_completed, ''MM/DD/YYYY'') || '' - ('' || event_title || '')''
                       ) ) as event_title,
                       
                t.EVENT_DESCR,
                t.SORT_ORDER,
                t.EVENT_FREQUENCY,
                t.PW_EVENT_TYPE_ID,
                t.PAT_PW_EVENT_ID
       
       FROM tbicds.PAT_PW_EVENT t
       WHERE t.patient_id = :PATIENT_ID 
       and t.pat_pw_id = :PAT_PW_ID
       ORDER BY nvl(t.DATE_COMPLETED,t.DATE_SCHEDULED) ASC';
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --open the result set
     open rs 
      for v_vSQL
    using pi_vPatientID,
          pi_nPatPWID;
    
    commit;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

/*****
Gets all pathway events associated with a patient
******/
procedure GetPatPWEventRS (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_vPatientID             in varchar2,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2,
      rs                        out RetRefCursor
      )
is

     v_vSQL varchar2(1500) :=
       
        'SELECT  t.PATIENT_ID,	
                t.STATUS,
                t.DATE_SCHEDULED,
                t.DATE_STARTED,
                t.DATE_COMPLETED,
                t.REGION_ID,
                t.SITE_ID,
                t.PAT_PW_ID,
                t.PLAN_TEXT,
                t.PLAN_PAT_PW_EVENT_ID,
               
                decode(trim(event_title),
                                 null,

                decode(t.date_completed,
                       null,
                       '''' || to_char(date_scheduled, ''MM/DD/YYYY'' ), 
                       '''' || to_char(date_completed, ''MM/DD/YYYY'')
                       ),
                       
                decode(t.date_completed,
                       null,
                       '''' || to_char(date_scheduled, ''MM/DD/YYYY'') || '' - ('' || event_title || '')'', 
                       '''' || to_char(date_completed, ''MM/DD/YYYY'') || '' - ('' || event_title || '')''
                       ) ) as event_title,
       
                t.EVENT_DESCR,
                t.SORT_ORDER,
                t.EVENT_FREQUENCY,
                t.PW_EVENT_TYPE_ID,
                t.PAT_PW_EVENT_ID

       FROM tbicds.PAT_PW_EVENT t
       WHERE t.patient_id = :PATIENT_ID 
         ORDER BY t.DATE_SCHEDULED ASC';
       
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --open the result set
     open rs 
      for v_vSQL
    using pi_vPatientID;
    
    commit;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

/*****
Gets one pathway event associated with a patient
******/
procedure GetPatPWEventRS (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_vPatientID             in varchar2,
      pi_nPatPWEventID          in number,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2,
      rs                        out RetRefCursor
      )
is

     v_vSQL varchar2(1500) :=
       
        'SELECT t.PATIENT_ID,	
                t.STATUS,
                t.DATE_SCHEDULED,
                t.DATE_STARTED,
                t.DATE_COMPLETED,
                t.REGION_ID,
                t.SITE_ID,
                t.PAT_PW_ID,
                t.event_title,
                t.EVENT_DESCR,
                t.SORT_ORDER,
                t.EVENT_FREQUENCY,
                t.PW_EVENT_TYPE_ID,
                t.PAT_PW_EVENT_ID,
                t.PLAN_PAT_PW_EVENT_ID,
                t.PLAN_TEXT,
                t.NOTE_TEXT,
                t.TIU_DFN,
                t.TIU_DUZ,
                t.TIU_ENCOUNTER_ID,
                t.TIU_CONSULT_IEN,
                t.TIU_COSIGNER_DUZ,
                t.TIU_NOTE_TITLE_IEN,
                t.TIU_NOTE,
                t.TIU_CLINIC_ID,
                t.TIU_DATE_SUBMITTED,
                t.TIU_ENCOUNTER_TYPE,
                t.TIU_NOTE_TITLE,
                t.TIU_LOCATION,
                t.TIU_COSIGNER,
                t.TIU_NOTE_ID,
                t.TIU_CONSULT
                
       FROM tbicds.PAT_PW_EVENT t
       WHERE t.patient_id = :PATIENT_ID 
         and t.pat_pw_event_id = :PAT_PW_EVENT_ID 
         ORDER BY t.DATE_SCHEDULED ASC';
       
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --open the result set
     open rs 
      for v_vSQL
    using pi_vPatientID,
          pi_nPatPWEventID;
    
    commit;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;




/*****
Gets all pathway event modules associated 
with a specific patient pathway event
******/
procedure GetPatPWEventModuleRS (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_vPatientID             in varchar2,
      pi_nPatPWID               in number,
      pi_nPatPWEventID             in number,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2,
      rs                        out RetRefCursor
      )
is

     v_vSQL varchar2(1000) :=
      'SELECT t.*,
              t3.module, 
              t3.description, 
              t3.module_alt_lang, 
              t3.description_alt_lang, 
              t3.acronym, 
              t3.score_logic,
              
              (select nvl(tt.status,0) from
                pat_pw_event tt
                where tt.pat_pw_event_id = :PAT_PW_EVENT_ID) 
                as event_status     
              
       FROM tbicds.PAT_PW_EVENT_MODULE t,
            tbicds.intake_module t3 
            
       WHERE t.patient_id = :PATIENT_ID 
         and t.PAT_PW_ID  = :PAT_PW_ID 
         and t.PAT_PW_EVENT_ID = :PW_EVENT_ID 
         
         and t3.mid = t.mid
         
       ORDER BY t.sort_order ASC';
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --open the result set
     open rs 
      for v_vSQL
    using pi_nPatPWEventID,
          pi_vPatientID,
          pi_nPatPWID,
          pi_nPatPWEventID;
    
    commit;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

/*get all modules for a patient pw event*/
procedure GetPatPWEventModuleRS (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_vPatientID             in varchar2,
      pi_nPatPWEventID             in number,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2,
      rs                        out RetRefCursor
      )
is

     v_vSQL varchar2(1000) :=
      'SELECT t.*,
              t3.module, 
              t3.description, 
              t3.module_alt_lang, 
              t3.description_alt_lang, 
              t3.acronym, 
              t3.score_logic   
              
       FROM tbicds.PAT_PW_EVENT_MODULE t,
            tbicds.intake_module t3 
            
       WHERE t.patient_id = :PATIENT_ID 
         and t.PAT_PW_EVENT_ID = :PW_EVENT_ID 
         
         and t3.mid = t.mid
         
       ORDER BY t.sort_order ASC';
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --open the result set
     open rs 
      for v_vSQL
    using pi_vPatientID,
          pi_nPatPWEventID;
    
    commit;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

/*****
Gets all pathway event modules associated with a patient 
******/
procedure GetPatPWEventModuleRS (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_vPatientID             in varchar2,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2,
      rs                        out RetRefCursor
      )
is

     v_vSQL varchar2(1000) :=
       'SELECT t.*, 
              
              t3.module, 
              t3.description, 
              t3.module_alt_lang, 
              t3.description_alt_lang, 
              t3.acronym, 
              t3.score_logic   
              
       FROM tbicds.PAT_PW_EVENT_MODULE t,
            tbicds.intake_module t3 
            
       WHERE t.patient_id = :PATIENT_ID 
         
         and t3.mid = t.mid
         
       ORDER BY t.pat_pw_id, t.pat_pw_event_id, t.sort_order ASC';
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --open the result set
     open rs 
      for v_vSQL
    using pi_vPatientID;
    
    commit;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

-------------------------------TODO: operations--------------------------------

procedure ApplyDefaultPathway (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      
      pi_vPatientID             in varchar2,
      pi_nRegionID              in number,
      pi_nSiteID                in number,
      
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2
      )
is
     v_nPatPWID   number := 0;
     v_nPWID      number := 0;
     v_nCPAID      number := 0;
     
     v_vSQL varchar2(4000) := '';

     v_vPWSelectSQL varchar2(1000) :=
        'select t.*
           from CPA_PW t
          where t.PW_ID = :PW_ID';  
          
     v_vPWEventSelectSQL varchar2(1000) :=
        'select t.*
           from CPA_PW_EVENT t 
          where t.PW_ID = :PW_ID
          order by sort_order asc';   
          
      v_vPWEventModuleSelectSQL varchar2(1000) :=
        'select t.*
           from CPA_PW_EVENT_MODULE t 
          where t.PW_ID = :PW_ID
          order by sort_order asc';    
          
      v_rsPWSelect  refCursor;
      v_rsPWEventSelect  refCursor;
      v_rsPWEventModuleSelect  refCursor;    
               
      v_recPWSelect  tbicds.cpa_pw%ROWTYPE;
      v_recPWEventSelect tbicds.cpa_pw_event%ROWTYPE;
      v_recPWEventModuleSelect tbicds.cpa_pw_event_module%ROWTYPE;
               
      v_nPatPWEventID number := 0;
      v_nCount number := 0;
      
      v_dtSched date := null;
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --the the one and only open pathway for this site
    select pw_id, cpa_id into v_nPWID, v_nCPAID from cpa_pw t
    where t.pw_type_id = 3
    and t.region_id = pi_nRegionID
    and t.site_id = pi_nSiteID 
    and rownum = 1;    
    
    --only apply if it doesnt exist!
    select count(*) into v_nCount 
    from pat_pw
    where patient_id = pi_vPatientID;
    
    if v_nCount > 0 then
      return;
    end if;
  
     --open the select recordsets
    open v_rsPWSelect for v_vPWSelectSQL using v_nPWID;
    open v_rsPWEventSelect for v_vPWEventSelectSQL using v_nPWID;
    open v_rsPWEventModuleSelect for v_vPWEventModuleSelectSQL using v_nPWID;
    
    --get a new patient pathway id
    select SEQPATPWID.nextval into v_nPatPWID from dual;
    
    --loop and copy pathway, should only be onw record
    LOOP
         v_recPWSelect := null;
         FETCH v_rsPWSelect INTO v_recPWSelect;
         EXIT WHEN NOT v_rsPWSelect%found;
  
         --add a record to the patient pathway table
         insert into tbicds.pat_pw
                (PATIENT_ID,
                 DATE_CREATED,
                 CREATED_BY,
                 STATUS,
                 PW_TYPE_ID,
                 REGION_ID,
                 SITE_ID,
                 PAT_PW_ID,
                 PW_TITLE,
                 PW_DESCR,
                 PW_ID,
                 CPA_ID)
         values(pi_vPatientID,
                sysdate,
                pi_nUserID,
                1, --1 = open
                v_recPWSelect.pw_type_id,
                v_recPWSelect.region_id,
                v_recPWSelect.site_id,
                v_nPatPWID,
                v_recPWSelect.pw_title,
                v_recPWSelect.pw_descr,
                v_nPWID,
                v_nCPAID);         
         
         commit;                             

    END LOOP;
    CLOSE v_rsPWSelect;
    
    --get a new patient pathway id
    select SEQPATPWEVENTID.nextval into v_nPatPWEventID from dual;
    v_dtSched := sysdate;
          
    --loop and copy pathway event
    LOOP
         v_recPWEventSelect := null;
         FETCH v_rsPWEventSelect INTO v_recPWEventSelect;
         EXIT WHEN NOT v_rsPWEventSelect%found;
         
         
         
         insert into pat_pw_event
                (PATIENT_ID,
                 STATUS,
                 DATE_SCHEDULED,
                 DATE_STARTED,
                 DATE_COMPLETED,
                 REGION_ID,
                 SITE_ID,
                 PAT_PW_ID,
                 EVENT_TITLE,
                 EVENT_DESCR,
                 SORT_ORDER,
                 EVENT_FREQUENCY,
                 PW_EVENT_TYPE_ID,
                 PAT_PW_EVENT_ID,
                 CREATED_BY)
        values(pi_vPatientID,
               0,
               v_dtSched,
               null,
               null,
               v_recPWEventSelect.REGION_ID,
               v_recPWEventSelect.SITE_ID,
               v_nPatPWID,
               --v_recPWEventSelect.EVENT_TITLE,
               --to_char(v_dtSched),
               '', --no title for default pathway
               v_recPWEventSelect.EVENT_DESCR,
               v_recPWEventSelect.SORT_ORDER,
               v_recPWEventSelect.EVENT_FREQUENCY,
               v_recPWEventSelect.PW_EVENT_TYPE_ID,
               v_nPatPWEventID,
               pi_nUserID);
        
        commit;                  

    END LOOP;
    CLOSE v_rsPWEventSelect;
        
    --no modules for open pathway
    
    commit;

exception
    when others then
      rollback;
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

/*
merge pathway events and modules
*/
procedure MergePathwayEvents (
      pi_vPatientID             in varchar2,
      pi_nPatPWID               in number,
      po_nMerged                out number
      )
is
      v_rsPatPWEvent  refCursor; 
      v_recPatPWEvent tbicds.pat_pw_event%ROWTYPE;
      v_nCount number := 0;
begin

    po_nMerged := 0;
       
    --update the base pathway (the first selected), 
    --move all mids from others matching freq and delete
    --the merged events and modules
    --MERGED = 1 moves to MERGED = 2
    open v_rsPatPWEvent for
    
          select * from PAT_PW_EVENT t 
          where pat_pw_id = pi_nPatPWID
          and MERGED = 1
          and patient_id = pi_vPatientID
          order by event_frequency asc; 
          
     LOOP
         --fetch a record for work   
         v_recPatPWEvent := null;
         FETCH v_rsPatPWEvent INTO v_recPatPWEvent;
         EXIT WHEN NOT v_rsPatPWEvent%found;
         
         begin
           
           insert into pat_pw_event_module
                   (PATIENT_ID,
                    MID,
                    STATUS,
                    REGION_ID,
                    SITE_ID,
                    DATE_STARTED,
                    DATE_COMPLETED,
                    PAT_PW_ID,
                    INTAKE_ID,
                    MODULE_FOR,
                    SORT_ORDER,
                    PAT_PW_EVENT_ID,
                    DATE_SCHEDULED,
                    merged,
                    PW_EVENT_MODULE_ID) 
         
           (select pem.PATIENT_ID,
                   pem.MID,
                   pem.STATUS,
                   pem.REGION_ID,
                   pem.SITE_ID,
                   pem.DATE_STARTED,
                   pem.DATE_COMPLETED,
                   pem.PAT_PW_ID,
                   pem.INTAKE_ID,
                   pem.MODULE_FOR,
                   9999 as SORT_ORDER,
                   v_recPatPWEvent.PAT_PW_EVENT_ID as PAT_PW_EVENT_ID,
                   pem.DATE_SCHEDULED,
                   1 as merged,
                   pem.PW_EVENT_MODULE_ID
                   
              from pat_pw_event_module pem
              
             where pem.pat_pw_id = pi_nPatPWID
             
               and pem.patient_id = pi_vPatientID
               
               and pem.pat_pw_event_id 
                   in (select pat_pw_event_id
                         from pat_pw_event ppe
                        where ppe.pat_pw_id = pi_nPatPWID
                          and ppe.patient_id = pi_vPatientID
                          and ppe.MERGED = 0 
                          and ppe.event_frequency = v_recPatPWEvent.event_frequency
                       ));
             commit;                   
       
           exception
           when others then 
             --dupes wont go but thats ok so press on
             null;
           end;
           
           
           --delete all dupes
           delete 
           from PAT_PW_EVENT_MODULE a
           where pat_pw_id = pi_nPatPWID
             and patient_id = pi_vPatientID
             and PAT_PW_EVENT_ID = v_recPatPWEvent.PAT_PW_EVENT_ID
             and a.rowid > 
             any 
               (select b.rowid from PAT_PW_EVENT_MODULE b
                 where pat_pw_id = pi_nPatPWID
                 and patient_id = pi_vPatientID
                 and PAT_PW_EVENT_ID = v_recPatPWEvent.PAT_PW_EVENT_ID
                 and a.mid = b.mid);
      
     END LOOP;
    
     --first pass done so now delete processed records
     --delete the modules merged
     delete from pat_pw_event_module t
     where t.pat_pw_id = pi_nPatPWID
     and patient_id = pi_vPatientID
     and pat_pw_event_id in 
          (select pat_pw_event_id
            from pat_pw_event
           where pat_pw_id = pi_nPatPWID
             and patient_id = pi_vPatientID
             and MERGED = 0 
             and event_frequency in
             ( select event_frequency
               from pat_pw_event
               where MERGED = 1 
               and pat_pw_id = pi_nPatPWID
               and patient_id = pi_vPatientID));
      commit;
      
      --delete the events merged 
      delete from pat_pw_event t
      where t.pat_pw_id = pi_nPatPWID
      and patient_id = pi_vPatientID
      and pat_pw_event_id in 
          (select pat_pw_event_id
            from pat_pw_event
           where pat_pw_id = pi_nPatPWID
             and patient_id = pi_vPatientID
             and MERGED = 0 
             and event_frequency in
             ( select event_frequency
               from pat_pw_event
               where MERGED = 1 
               and pat_pw_id = pi_nPatPWID
               and patient_id = pi_vPatientID));
       commit;
       
       --now mark the merged base to 2
       --2 means all merged so dont have to process anymore
       update pat_pw_event t
       set t.MERGED = 2
       where t.pat_pw_id = pi_nPatPWID
       and patient_id = pi_vPatientID
       and t.MERGED = 1;
       commit;
       
       --now update for the next set to merge
       update PAT_PW_EVENT pe
       set pe.merged = 1 
       where pe.pat_pw_id = pi_nPatPWID
         and pe.patient_id = pi_vPatientID
          
         and pe.event_frequency in 
             (select t.event_frequency 
                 from PAT_PW_EVENT t 
                where t.pat_pw_id = pi_nPatPWID
                  and t.patient_id = pi_vPatientID
                  and t.merged != 2
                 
                  and (select count(*) 
                         from PAT_PW_EVENT t2 
                        where t2.pat_pw_id = pi_nPatPWID
                          and t2.patient_id = pi_vPatientID
                          and t2.merged != 2
                          and t2.event_frequency = t.event_frequency 
                     group by t2.event_frequency) > 1 )
         
         and rownum = 1;
         commit;
  
       
       --do we still have work to do?
       select count(*) into v_nCount
       from pat_pw_event t 
       where t.pat_pw_id = pi_nPatPWID
       and t.patient_id = pi_vPatientID
       and t.MERGED = 1;
       
       if v_nCount < 1 then
         po_nMerged := 1; --all done
       else
         po_nMerged := 0; --more to do
       end if;
       
exception
    when others then
      rollback;
      po_nMerged := 1;
      
end;

/******
apply a pathway/schedule to a patient
******/  
procedure ApplyPathway (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_vPatientID             in varchar2,
      
      pi_vPWIDs                 in varchar2,
      pi_vPWTitle               in varchar2,
      pi_dtBaseline             in date,
      
      po_nPatPWID               out number,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2
      )
is
     v_nPatPWID   number := 0;
     
     v_vSQL varchar2(4000) := '';

     v_vPWSelectSQL varchar2(1000) :=
        'select t.*
           from CPA_PW t
          where t.PW_ID = :PW_ID';  
          
     v_vPWEventSelectSQL varchar2(1000) :=
        'select t.*
           from CPA_PW_EVENT t 
          where t.PW_ID = :PW_ID 
          order by sort_order asc';  
       
          
      v_vPWEventModuleSelectSQL varchar2(1000) :=
        'select t.*
           from CPA_PW_EVENT_MODULE t 
          where t.PW_ID = :PW_ID
          order by sort_order asc';    
          
      v_rsPWSelect  refCursor;
      v_rsPWEventSelect  refCursor;
      v_rsPWEventModuleSelect  refCursor;  
      
      v_rsPatPWEvent  refCursor; 
      v_recPatPWEvent tbicds.pat_pw_event%ROWTYPE;
       
      
      v_recPWSelect  tbicds.cpa_pw%ROWTYPE;
      v_recPWEventSelect tbicds.cpa_pw_event%ROWTYPE;
      v_recPWEventModuleSelect tbicds.cpa_pw_event_module%ROWTYPE;
    
      v_nCPAID number := 0;  
      v_nPWID number := 0;       
      v_nPatPWEventID number := 0;  
      
      v_dtScheduled date := null;
      v_nEventFrequency number := 0;
      v_nIndex number := 0;
      v_vPiece varchar2(4000) := '';
      
      v_nPWRecordInserted number := 0;
     
      --fisrt pw_id is always the master for combining
      v_nMaster number := 0;
      v_nMerged number := 0;
      
      v_nEventModuleID number := 0;
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    po_nPatPWID := 0;
    v_nPWID := 0;
    
    
    --get a new patient pathway id
    select SEQPATPWID.nextval into v_nPatPWID from dual;
      
    loop
      
      --get one pathway id   
      v_vPiece := pck_common.GetPiece(pi_vPWIDs, ',' , v_nIndex);
      EXIT WHEN v_vPiece is null;
      
      --bump the index
      v_nIndex := v_nIndex + 1;
    
      --convert the piece to a PW_ID number
      v_nPWID := to_number(v_vPiece);
      
      --get the cpa id
      select cpa_id into v_nCPAID
      from cpa_pw cp
      where cp.pw_id = v_nPWID;      
    
      --open the select recordsets
      open v_rsPWSelect for v_vPWSelectSQL using v_nPWID;
      open v_rsPWEventSelect for v_vPWEventSelectSQL using v_nPWID;
       
      --first record in list is used for pw insert
      v_recPWSelect := null;
      FETCH v_rsPWSelect INTO v_recPWSelect;
      if v_nPWRecordInserted = 0 then 
        
         --add a record to the patient pathway table
         insert into tbicds.pat_pw
                  (PATIENT_ID,
                   DATE_CREATED,
                   CREATED_BY,
                   STATUS,
                   PW_TYPE_ID,
                   REGION_ID,
                   SITE_ID,
                   PAT_PW_ID,
                   PW_TITLE,
                   PW_DESCR,
                   CPA_ID,
                   PW_ID)
           values(pi_vPatientID,
                  sysdate,
                  pi_nUserID,
                  1, --1 = open
                  v_recPWSelect.pw_type_id,
                  v_recPWSelect.region_id,
                  v_recPWSelect.site_id,
                  v_nPatPWID,
                  pi_vPWTitle,
                  v_recPWSelect.pw_descr,
                  v_nCPAID,
                  v_nPWID);         
           
           commit;           
           v_nPWRecordInserted := 1;
           v_nMaster := 1;
      else
           v_nMaster := 0;
      end if;
      CLOSE v_rsPWSelect;
      
      --loop and copy pathway event
      LOOP
           --fetch a record for work   
           v_recPWEventSelect := null;
           FETCH v_rsPWEventSelect INTO v_recPWEventSelect;
           EXIT WHEN NOT v_rsPWEventSelect%found;
           
           --get a new patient pathway id
           select SEQPATPWEVENTID.nextval into v_nPatPWEventID from dual;
           
           --calculate the estimated schedule date based 
           --on the baseline and interval
           v_dtScheduled := null;
           v_nEventFrequency := v_recPWEventSelect.EVENT_FREQUENCY;
           if v_nEventFrequency > 0 then
              v_dtScheduled := pi_dtBaseline + v_nEventFrequency;
           else
              v_dtScheduled := pi_dtBaseline;
           end if;
           
           --insert the event
           insert into pat_pw_event
                  (PATIENT_ID,
                   STATUS,
                   DATE_SCHEDULED,
                   DATE_STARTED,
                   DATE_COMPLETED,
                   REGION_ID,
                   SITE_ID,
                   PAT_PW_ID,
                   EVENT_TITLE,
                   EVENT_DESCR,
                   SORT_ORDER,
                   EVENT_FREQUENCY,
                   PW_EVENT_TYPE_ID,
                   PAT_PW_EVENT_ID,
                   MERGED,
                   CREATED_BY)
          values(pi_vPatientID,
                 0,
                 v_dtScheduled,
                 null,
                 null,
                 v_recPWEventSelect.REGION_ID,
                 v_recPWEventSelect.SITE_ID,
                 v_nPatPWID,
                 v_recPWEventSelect.EVENT_TITLE,
                 v_recPWEventSelect.EVENT_DESCR,
                 v_recPWEventSelect.SORT_ORDER,
                 v_recPWEventSelect.EVENT_FREQUENCY,
                 v_recPWEventSelect.PW_EVENT_TYPE_ID,
                 v_nPatPWEventID,
                 v_nMaster,
                 pi_nUserID);
          
         commit;            
                   
         --loop and copy pathway event module
         open v_rsPWEventModuleSelect for v_vPWEventModuleSelectSQL using v_nPWID;
         LOOP
           v_recPWEventModuleSelect := null;
           FETCH v_rsPWEventModuleSelect INTO v_recPWEventModuleSelect;
           EXIT WHEN NOT v_rsPWEventModuleSelect%found;
    
         if v_recPWEventModuleSelect.PW_ID = v_recPWEventSelect.PW_ID
             and v_recPWEventModuleSelect.PW_EVENT_ID = v_recPWEventSelect.PW_EVENT_ID
         then   
           
           select SEQPATPWEVENTMODULEID.Nextval 
           into v_nEventModuleID from dual;
              
           insert into pat_pw_event_module
                  ( PATIENT_ID,
                    MID,
                    STATUS,
                    REGION_ID,
                    SITE_ID,
                    DATE_STARTED,
                    DATE_COMPLETED,
                    PAT_PW_ID,
                    INTAKE_ID,
                    MODULE_FOR,
                    SORT_ORDER,
                    DATE_SCHEDULED,
                    PAT_PW_EVENT_ID,
                    PW_EVENT_MODULE_ID)
           values( pi_vPatientID,
                  v_recPWEventModuleSelect.MID,
                  0,
                  v_recPWEventModuleSelect.REGION_ID,
                  v_recPWEventModuleSelect.SITE_ID,
                  null,
                  null,
                  v_nPatPWID,
                  null,
                  v_recPWEventModuleSelect.MODULE_FOR,
                  v_recPWEventModuleSelect.SORT_ORDER,
                  v_dtScheduled,
                  v_nPatPWEventID,
                  v_nEventModuleID);

           commit;  
         
         end if;
         
         END LOOP;
         CLOSE v_rsPWEventModuleSelect;
         
     
      END LOOP;
      --CLOSE v_rsPWEventSelect;
    
    END LOOP;   
    
    --merge pathway events
    v_nMerged := 0;
    LOOP
    EXIT WHEN v_nMerged = 1;
    
        MergePathwayEvents(pi_vPatientID, v_nPatPWID, v_nMerged);
    
    END LOOP;
        
    --return the patient pathway id
    po_nPatPWID := v_nPatPWID;
           
    commit;

exception
    when others then
      rollback;
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

/***
Insert pathway event
***/
procedure InsertPathwayEvent (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_vPatientID             in varchar2,
      pi_nPWID                  in number,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2,
      rs                        out RetRefCursor
      )
is

     v_vSQL varchar2(1000) :=
       'SELECT t.*, 
              
              t2.module_for, 
              t2.sort_order,
              
              t3.module, 
              t3.description, 
              t3.module_alt_lang, 
              t3.description_alt_lang, 
              t3.acronym, 
              t3.score_logic   
              
       FROM tbicds.PAT_PW_EVENT_MODULE t,
            tbicds.CPA_PW_EVENT_MODULE t2,
            tbicds.intake_module t3 
            
       WHERE t.patient_id = :PATIENT_ID 
         
         and t2.PW_ID = t.PW_ID
         and t2.PW_EVENT_ID = t.PW_EVENT_ID 
         and t2.MID = t.MID
         
         and t3.mid = t.mid
         
       ORDER BY t.cpa_id, t.pw_id, t.pw_event_id, t2.sort_order ASC';
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --open the result set
     open rs 
      for v_vSQL
    using pi_vPatientID;
    
    commit;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

/***
Insert pat pw event modules
***/
procedure InsertPatPWEventModules (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_vPatientID             in varchar2,
      pi_nPatPWEventID          in number,
      pi_vMIDS                  in varchar2,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2
      )
is

      v_nIndex number := 0;
      v_vPiece varchar2(4000) := '';   
      v_nMID number := 0;   
   
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
     loop
      
      --get one mid   
      v_vPiece := pck_common.GetPiece(pi_vMIDs, ',' , v_nIndex);
      EXIT WHEN v_vPiece is null;
            
      begin
        
           --get the mid and bump the index
           v_nMID := to_number(v_vPiece);
           v_nIndex := v_nIndex +1;
           
           insert into 
           pat_pw_event_module(PATIENT_ID,
                                    MID,
                                    STATUS,
                                    REGION_ID,
                                    SITE_ID,
                                    DATE_STARTED,
                                    DATE_COMPLETED,
                                    PAT_PW_ID,
                                    INTAKE_ID,
                                    MODULE_FOR,
                                    SORT_ORDER,
                                    PAT_PW_EVENT_ID,
                                    DATE_SCHEDULED)
            values(pi_vPatientID,
                   
                   v_nMID,
                   
                   0,
                   
                   (select REGION_ID 
                    from pat_pw_event 
                    where pat_pw_event_id = pi_nPatPWEventID),
                   
                    (select SITE_ID 
                    from pat_pw_event 
                    where pat_pw_event_id = pi_nPatPWEventID),
                    
                    null,
                    null,
                   
                   (select PAT_PW_ID 
                    from pat_pw_event 
                    where pat_pw_event_id = pi_nPatPWEventID),
                   
                    null,
                    1,
                    9999999,
                    pi_nPatPWEventID,
                    sysdate);
                                    
           commit;                                 
           
      
      exception
      when others then
           null;
      end;
      
     end loop;
   
    
    commit;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

/***
Update the pw event module list
****/
procedure UpdatePatPWEventModules (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_vPatientID             in varchar2,
      pi_nPatPWEventID          in number,
      
      pi_vEventTitle            in varchar2,
      pi_dtEventDate            in date,
      pi_nAdjustFuture          in number,
      
      pi_vMIDS                  in varchar2,
      pi_vModuleFors            in varchar2,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2
      )
is

      v_nIndex number := 0;
      v_vMIDPiece varchar2(4000) := ''; 
      v_vFORPiece varchar2(4000) := '';  
      v_nMID number := 0;   
      v_nModuleFor number := 2;
      
      v_dtOldScheduled date := null;
      v_nPatPWID number := 0;
      v_nDays number := 0;
      v_nEventModuleID number := 0;
   
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    v_nEventModuleID := 0;
    
    --keep the original date for calculations
    select date_scheduled, pat_pw_id 
    into v_dtOldScheduled, v_nPatPWID
    from pat_pw_event 
    where pat_pw_event_id = pi_nPatPWEventID
    and patient_id = pi_vPatientID;
    
    --update title and date
    update pat_pw_event 
    set event_title = pi_vEventTitle,
        date_scheduled = pi_dtEventDate
    where pat_pw_event_id = pi_nPatPWEventID
    and patient_id = pi_vPatientID;
    commit;
        
    --get the number of days this date is different    
    if pi_nAdjustFuture = 1 then
       
       --get the date diff
       SELECT trunc(pi_dtEventDate)- (v_dtOldScheduled) 
         into v_nDays 
         FROM DUAL;
        
       --update all events after this event
       update pat_pw_event pe
          set pe.date_scheduled =  pe.date_scheduled + v_nDays
        where patient_id = pi_vPatientID
          and pe.pat_pw_event_id > pi_nPatPWEventID
          and pe.pat_pw_id = pat_pw_id;
       commit;         
      
    end if;
    
    --remove all the current modules, will update below
    delete from pat_pw_event_module pem 
    where pem.pat_pw_event_id = pi_nPatPWEventID
    and pem.patient_id = pi_vPatientID 
    and pem.intake_id is null;
    
    --loop and add modules
     loop
      
      --get one mid   
      v_vMIDPiece := pck_common.GetPiece(pi_vMIDs, ',' , v_nIndex);
      
      --get one module for
      v_vFORPiece := pck_common.GetPiece(pi_vModuleFors, ',' , v_nIndex);
      
      EXIT WHEN v_vMIDPiece is null;
            
      begin
        
           --get the mid and bump the index
           v_nMID := to_number(v_vMIDPiece);
           v_nModuleFor := to_number(v_vFORPiece);
           v_nIndex := v_nIndex +1;
           
           select SEQPATPWEVENTMODULEID.Nextval 
           into v_nEventModuleID from dual;
           
           insert into 
           pat_pw_event_module(PATIENT_ID,
                               MID,
                               STATUS,
                               REGION_ID,
                               SITE_ID,
                               DATE_STARTED,
                               DATE_COMPLETED,
                               PAT_PW_ID,
                               INTAKE_ID,
                               MODULE_FOR,
                               SORT_ORDER,
                               PAT_PW_EVENT_ID,
                               DATE_SCHEDULED,
                               PW_EVENT_MODULE_ID)
            values(pi_vPatientID,
                   
                   v_nMID,
                   
                   0,--status
                   
                   (select REGION_ID 
                    from pat_pw_event 
                    where pat_pw_event_id = pi_nPatPWEventID),
                   
                    (select SITE_ID 
                    from pat_pw_event 
                    where pat_pw_event_id = pi_nPatPWEventID),
                    
                    null, --date started
                    
                    null, --date completed
                   
                   (select PAT_PW_ID 
                    from pat_pw_event 
                    where pat_pw_event_id = pi_nPatPWEventID),
                   
                    null, --intake_id
                    
                    v_nModuleFor, --module for
                    
                    (select nvl(max(sort_order), 0) + 1 
                       from pat_pw_event_module 
                      where pat_pw_event_id = pi_nPatPWEventID),
                      
                    pi_nPatPWEventID,
                    
                    sysdate,
                    
                    v_nEventModuleID);
                                    
           commit;                                 
           
      
      exception
      when others then
           null;
      end;
      
     end loop;
   
    
    commit;

exception
    when others then
      rollback;
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

/*update a pathway event module*/
procedure UpdatePathwayEventModule (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_vPatientID             in varchar2,
      pi_nPWID                  in number,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2,
      rs                        out RetRefCursor
      )
is

     v_vSQL varchar2(1000) :=
       'SELECT t.*, 
              
              t2.module_for, 
              t2.sort_order,
              
              t3.module, 
              t3.description, 
              t3.module_alt_lang, 
              t3.description_alt_lang, 
              t3.acronym, 
              t3.score_logic   
              
       FROM tbicds.PAT_PW_EVENT_MODULE t,
            tbicds.CPA_PW_EVENT_MODULE t2,
            tbicds.intake_module t3 
            
       WHERE t.patient_id = :PATIENT_ID 
         
         and t2.PW_ID = t.PW_ID
         and t2.PW_EVENT_ID = t.PW_EVENT_ID 
         and t2.MID = t.MID
         
         and t3.mid = t.mid
         
       ORDER BY t.cpa_id, t.pw_id, t.pw_event_id, t2.sort_order ASC';
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --open the result set
     open rs 
      for v_vSQL
    using pi_vPatientID;
    
    commit;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

/*****
Updates Patient Event Module status after a module is completed
*****/
procedure UpdatePatPWEventModule (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_vPatientID             in varchar2,
      pi_nPWID                  in number,
      pi_nPatPWEvtID            in number,
      pi_nMID                   in number,
      pi_nIntakeID              in number,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2
      )
is

     v_vSQL varchar2(1000) := '
        update PAT_PW_EVENT_MODULE t
           set t.status = 1,
               t.intake_id = :INTAKE_ID,
               t.date_started = sysdate,
               t.date_completed = sysdate,
               t.completed_by = :USER_ID
         where t.patient_id = :PATIENT_ID
           and t.pat_pw_id = :PAT_PW_ID
           and t.pat_pw_event_id = :PAT_PW_EVT_ID
           and t.mid = :MID';

begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --execute the SQL
    execute immediate v_vSQL 
    using pi_nIntakeID, 
          pi_nUserID,
          pi_vPatientID,
          pi_nPWID,
          pi_nPatPWEvtID, 
          pi_nMID;
                 
    commit;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

/*update a patient pathway event module*/
procedure UpdatePatPWEventModule (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_vPatientID             in varchar2,
      pi_nPWID                  in number,
      pi_nPatPWEvtID            in number,
      pi_nMID                   in number,
      pi_nIntakeID              in number,
      pi_nPwEvtModID              in number,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2
      )
is

     v_vSQL varchar2(1000) := '
        update PAT_PW_EVENT_MODULE t
           set t.status = 1,
               t.intake_id = :INTAKE_ID,
               t.date_started = sysdate,
               t.date_completed = sysdate,
               t.completed_by = :USER_ID
         where t.patient_id = :PATIENT_ID
           and t.pat_pw_id = :PAT_PW_ID
           and t.pat_pw_event_id = :PAT_PW_EVT_ID
           and t.mid = :MID
           and t.PW_EVENT_MODULE_ID = :PW_EVENT_MODULE_ID';

begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --execute the SQL
    execute immediate v_vSQL 
    using pi_nIntakeID, 
          pi_nUserID,
          pi_vPatientID,
          pi_nPWID,
          pi_nPatPWEvtID, 
          pi_nMID,
          pi_nPwEvtModID;
                 
    commit;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;


end PCK_CPA;
/

